/*
 * CollisionHandlerEventNode.h
 *
 * Created 8/9/2009 By Johnny Huynh
 *
 * Version 00.00.02 8/22/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef COLLISION_HANDLER_EVENT_NODE_H
 #define COLLISION_HANDLER_EVENT_NODE_H
 
 template <typename T> class CollisionHandlerEventNode;
 
 #include "nodePath.h"
 
 // For collision handling
 #include "eventHandler.h"
 #include "collisionNode.h"
 #include "collisionSphere.h"
 #include "collisionHandlerEvent.h"
 
 #include "global.h"
 #include "ObjectCollection.h"
 #include "referenceCount.h"
 #include "pointerTo.h"
 
 #include <vector>
 
 typedef CollisionHandlerEventNode<PRIMARY_TYPE> Collider;
 
 #define FROM_COLLISION_EVENT_NAME "From Collision"
 #define FROM_COLLISION_EVENT_FUNCTION &CollisionHandlerEventNode<T>::handle_in_pattern_collision
 
 #ifndef OBJECT_KEY_TAG
 #define OBJECT_KEY_TAG "Object Key"
 #endif // OBJECT_KEY_TAG
 
 #ifndef DAMAGE_OBJECT_KEY_TAG
 #define DAMAGE_OBJECT_KEY_TAG "Damage Key"
 #endif // DAMAGE_OBJECT_KEY_TAG
 
 /**
  * Class specification for class CollisionHandlerEventNode
  */
 template <typename T>
 class CollisionHandlerEventNode : public NodePath, public ReferenceCount
 {
 // Private Encapsulated Classes
 private:
    class CollisionEvent;
 
 // Private Static Functions
 private:
    static inline void handle_in_pattern_collision( const Event * e );
 
 // Data Members
 protected:
    //NodePath _collider; // NodePath of this CollisionNode
    PT(CollisionHandlerEvent) _collision_handler_Ptr; // destructor is virtual from being a subclass of TypedObject
    std::vector<PT(CollisionEvent)> _events; // contains all the events associated with this CollisionHandlerEventNode
    bool _is_from_collider; // true specifies that this collider can collider "into" other colliders
 
 // Special Purpose Functions
 protected:
    CollisionHandlerEventNode( const std::string& name, bool is_from_collider, bool initialize ); // for derived classes (i.e. sub-classes)
    CollisionHandlerEventNode( const CollisionHandlerEventNode<T>& chen, CollisionHandlerEvent* collision_handler_Ptr ); // for get_copy() function
 
 // Local Functions
 public:
    CollisionHandlerEventNode( const std::string& name, bool is_from_collider = true );
    CollisionHandlerEventNode( const CollisionHandlerEventNode<T>& chen );
    virtual ~CollisionHandlerEventNode();
    //virtual inline CollisionHandlerEventNode<T>& operator=( const CollisionHandlerEventNode<T>& chen );
    virtual inline bool add_event( const std::string& event_name = FROM_COLLISION_EVENT_NAME, EventHandler::EventFunction* event_function = FROM_COLLISION_EVENT_FUNCTION );
    virtual inline void clear_events();
    virtual inline PT(CollisionHandlerEventNode<T>) get_copy() const;
    virtual inline bool remove_events( const std::string& event_name );
    virtual inline void turn_on_from_collision();
    virtual inline void turn_off_from_collision();
    
    // To call CollisionNode functions
    inline CollisionNode* collision_node();
    
    // NodePath functions
    /*inline int get_key() { return _collider.get_key(); }
    inline std::string get_net_tag( const std::string& key ) { return _collider.get_net_tag( key ); }
    virtual inline void remove_node() 
    { 
        if ( !_collider.is_empty() ) 
        {
            CollisionHandlerEventNode<T>::detach_node(); 
            _collider.remove_node();
        } 
    }
    inline void hide() { _collider.hide(); }
    inline void show() { _collider.show(); }*/
    
    // CollisionHandlerEvent functions
    inline void add_again_pattern(string const &again_pattern = FROM_COLLISION_EVENT_NAME) { _collision_handler_Ptr->add_again_pattern( again_pattern );  }
    inline void add_in_pattern(string const &in_pattern = FROM_COLLISION_EVENT_NAME) { _collision_handler_Ptr->add_in_pattern( in_pattern );  }
    inline void add_out_pattern(string const &out_pattern = FROM_COLLISION_EVENT_NAME) { _collision_handler_Ptr->add_out_pattern( out_pattern );  }
    inline void clear() { _collision_handler_Ptr->clear(); }
    inline void clear_again_patterns() { _collision_handler_Ptr->clear_again_patterns(); }
    inline void clear_in_patterns() { _collision_handler_Ptr->clear_in_patterns(); }
    inline void clear_out_patterns() { _collision_handler_Ptr->clear_out_patterns(); }
    inline void flush() { _collision_handler_Ptr->flush(); }
    inline std::string get_again_pattern(int n) const { return _collision_handler_Ptr->get_again_pattern(n); }
    inline std::string get_in_pattern(int n) const { return _collision_handler_Ptr->get_in_pattern(n); }
    inline int get_num_again_patterns() const { return _collision_handler_Ptr->get_num_again_patterns(); }
    inline int get_num_in_patterns() const { return _collision_handler_Ptr->get_num_in_patterns(); }
    inline int get_num_out_patterns() const { return _collision_handler_Ptr->get_num_out_patterns(); }
    inline std::string get_out_pattern(int n) const { return _collision_handler_Ptr->get_out_pattern(n); }
    // these functions are deprecated
    //inline void set_again_pattern(string const &again_pattern)
    //                            { std::string collider_key( StringConversion::to_str( _collider.get_key() ) ); 
    //                              _collision_handler_Ptr->set_again_pattern( collider_key + "-" + again_pattern );  }
    //inline void set_in_pattern(string const &in_pattern)
    //                            { std::string collider_key( StringConversion::to_str( _collider.get_key() ) );
    //                              _collision_handler_Ptr->set_in_pattern( collider_key + "-" + in_pattern );  }
    //inline void set_out_pattern(string const &out_pattern) { _collision_handler_Ptr->set_out_pattern( out_pattern ); }
    //                            { std::string collider_key( StringConversion::to_str( _collider.get_key() ) );
    //                              _collision_handler_Ptr->set_out_pattern( collider_key + "-" + out_pattern );  }
    
    // overloaded functions (NodePath)
    virtual inline void reparent_to( NodePath& node_path );
    virtual inline void detach_node();
    static TypeHandle get_class_type() { return NodePath::get_class_type(); }
    static void init_type() { return NodePath::init_type(); }
	//static TypeHandle get_class_type() { return _type_handle; }
    //static void init_type() {
    //                            std::string template_type( typeid( T ).name() );
    //                            register_type(_type_handle, "CollisionHandlerEventNode<" + template_type + ">" );
    //                        }
 
 // Private Functions
 private:
    virtual inline void initialize_collision_handler();
 
 // Public Static Functions
 public:
    static inline bool add_global_event( const std::string& event_name, EventHandler::EventFunction* event_function );
    static inline bool remove_global_event( const std::string& event_name, EventHandler::EventFunction* event_function );
 
 // Private Encapsulated Classes
 private:
    class CollisionEvent : public ReferenceCount
    {
    // Data Members
    public:
        std::string _name;
        EventHandler::EventFunction* _function; // typedef void EventFunction(const Event *); // "eventHandler.h"
    
    // Local Functions
    public:
        CollisionEvent( const std::string& name, EventHandler::EventFunction* function ) : ReferenceCount(), _name( name ), _function( function ) { /*CollisionHandlerEventNode<T>::add_global_event( _name, _function );*/ }
        CollisionEvent( const CollisionEvent& ce ) : ReferenceCount(), _name( ce._name ), _function( ce._function ) {}
        ~CollisionEvent() { CollisionHandlerEventNode<T>::remove_global_event( _name, _function ); }
        inline CollisionEvent& operator=( const CollisionEvent& ce ) { _name = ce._name; _function = ce._function; return *this; }
    };
 };
 
 /** PRIVATE STATIC FUNCTIONS **/
 
 /**
  * handle_in_pattern_collision() handles the collision between two objects for an in-pattern collision.
  *
  * @param (const Event*) e
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::handle_in_pattern_collision( const Event * e )
 {
    //printf("Number of Parameters: %1d\n", e->get_num_parameters());
    nassertv( e->get_num_parameters() == 1 );
    TypedWritableReferenceCount * value( e->get_parameter(0).get_ptr() ); // pointer to CollisionEntry
    PT(CollisionEntry) collision_entry_Ptr( DCAST(CollisionEntry, value) ); 
    nassertv( collision_entry_Ptr != NULL );
    
    const std::string& from_object_key( collision_entry_Ptr->get_from_node_path().get_tag( "Object Key" ) );
    const std::string& into_object_key( collision_entry_Ptr->get_into_node_path().get_tag( "Object Key" ) );
    
    PT(Object<T>) from_object_Ptr( global::_world_Ptr->get_map_area()->get_object( from_object_key ) );
    PT(Object<T>) into_object_Ptr( global::_world_Ptr->get_map_area()->get_object( into_object_key ) );
    
    // if ( from_object_Ptr == NULL || into_object_Ptr == NULL )
    //  return; // a projectile object will destruct upon colliding, the second pass will yield a NULL pointer (from-into, into-from)
    nassertv( from_object_Ptr != NULL );
    nassertv( into_object_Ptr != NULL );
    
    // Set the contact normal
    //VECTOR3_TYPE contact_normal( from_object_Ptr->get_pos() - into_object_Ptr->get_pos() );
    collision_entry_Ptr->set_contact_normal( from_object_Ptr->get_pos() - into_object_Ptr->get_pos() );
    //VECTOR2_TYPE from_object_dir( Vector::get_xy_direction( from_object_Ptr->get_h() ) );
    //collision_entry_Ptr->set_contact_normal( VECTOR3_TYPE( from_object_dir.get_x(), from_object_dir.get_y(), ZERO ) );
    
    const std::string& from_dmg_object_key( collision_entry_Ptr->get_from_node_path().get_tag( "Damage Key" ) );
    const std::string& into_dmg_object_key( collision_entry_Ptr->get_into_node_path().get_tag( "Damage Key" ) );
    if ( from_dmg_object_key != "" )
    {
        PT(DamageObject<T>) from_dmg_object_Ptr = from_object_Ptr->get_damage_object( from_dmg_object_key );
        nassertv( from_dmg_object_Ptr != NULL );
        
        if ( into_dmg_object_key != "" ) // from_dmg_object_key != "" && into_dmg_object_key != ""
        {
            PT(DamageObject<T>) into_dmg_object_Ptr = into_object_Ptr->get_damage_object( into_dmg_object_key );
            nassertv( into_dmg_object_Ptr != NULL );
            from_dmg_object_Ptr->handle_from_collision( *into_dmg_object_Ptr, *collision_entry_Ptr );
            
            // reverse contact normal when DamageObject is collided into
            //collision_entry_Ptr->set_contact_normal( -contact_normal );
            into_dmg_object_Ptr->handle_into_collision( *from_dmg_object_Ptr, *collision_entry_Ptr );
        }
        else // from_dmg_object_key != "" && into_dmg_object_key == ""
        {
            from_dmg_object_Ptr->handle_from_collision( *into_object_Ptr, *collision_entry_Ptr );
            into_object_Ptr->handle_into_collision( *from_dmg_object_Ptr, *collision_entry_Ptr );
            
            // Simulation of locking on to target hit
            /*VECTOR3_TYPE reverse_contact_normal( into_object_Ptr->get_pos() - from_object_Ptr->get_pos() );
            VECTOR2_TYPE dir( reverse_contact_normal.get_x(), reverse_contact_normal.get_y() );
            Vector::normalize( dir );
            
            from_object_Ptr->set_h( Vector::get_angle_degrees(dir.get_x(), dir.get_y()) );*/
        }
    }
    else if ( into_dmg_object_key != "" ) // from_dmg_object_key == "" && into_dmg_object_key != ""
    {
        PT(DamageObject<T>) into_dmg_object_Ptr = into_object_Ptr->get_damage_object( into_dmg_object_key );
        nassertv( into_dmg_object_Ptr != NULL );
        
        // reverse contact normal when DamageObject is collided into
        //collision_entry_Ptr->set_contact_normal( -contact_normal );
        into_dmg_object_Ptr->handle_into_collision( *from_object_Ptr, *collision_entry_Ptr );
        from_object_Ptr->handle_from_collision( *into_dmg_object_Ptr, *collision_entry_Ptr );
    }
    else // from_dmg_object_key == "" && into_dmg_object_key == ""
    {
        from_object_Ptr->handle_from_collision( *into_object_Ptr, *collision_entry_Ptr );
        into_object_Ptr->handle_into_collision( *from_object_Ptr, *collision_entry_Ptr );
    }
    
    cerr << "Collision from " << from_object_key
         << " into " << into_object_key << "\n";
    //cerr << "Collision from " << collision_entry_Ptr->get_from_node_path().get_key() 
         //<< " into " << collision_entry_Ptr->get_into_node_path().get_key() << "\n";
 }
 
    /*//printf("Number of Parameters: %1d\n", e->get_num_parameters());
    nassertv( e->get_num_parameters() == 1 );
    TypedWritableReferenceCount * value( e->get_parameter(0).get_ptr() ); // pointer to CollisionEntry
    PT(CollisionEntry) collision_entry_Ptr( DCAST(CollisionEntry, value) ); 
    nassertv( collision_entry_Ptr != NULL );
    
    const std::string& from_object_key( collision_entry_Ptr->get_from_node_path().get_tag( "Object Key" ) );
    const std::string& into_object_key( collision_entry_Ptr->get_into_node_path().get_tag( "Object Key" ) );
    
    PT(Object<T>) from_object_Ptr( global::_world_Ptr->get_map_area()->get_object( from_object_key ) );
    PT(Object<T>) into_object_Ptr( global::_world_Ptr->get_map_area()->get_object( into_object_key ) );
    
    nassertv( from_object_Ptr != NULL );
    nassertv( into_object_Ptr != NULL );
    
    from_object_Ptr->handle_from_collision( *into_object_Ptr, *collision_entry_Ptr );
    into_object_Ptr->handle_into_collision( *from_object_Ptr, *collision_entry_Ptr );
    
    cerr << "Collision from " << from_object_key
         << " into " << into_object_key << "\n";
    //cerr << "Collision from " << collision_entry_Ptr->get_from_node_path().get_key() 
         //<< " into " << collision_entry_Ptr->get_into_node_path().get_key() << "\n";*/
 
 /** PROTECTED FUNCTIONS **/
 
 /**
  * Special Purpose Constructor - for subclasses of this class only
  * To activate a collider, use reparent_to() to attach this node to a NodePath.
  * To deactivate a collider, use detach_node().
  *
  * @param (const std::string&) name - name for this CollisionHandlerEventNode
  * @param (bool) initialize - if true, initialize this collider
  */
 template <typename T>
 CollisionHandlerEventNode<T>::CollisionHandlerEventNode( const std::string& name, bool is_from_collider, bool initialize )
                              : NodePath( new CollisionNode( name ) ),
                                ReferenceCount(),
                                _is_from_collider( is_from_collider )
 {
    if ( initialize )
        CollisionHandlerEventNode<T>::initialize_collision_handler();
 }
 
 /**
  * Special Purpose Copy Constructor
  */
 template <typename T>
 CollisionHandlerEventNode<T>::CollisionHandlerEventNode( const CollisionHandlerEventNode<T>& chen, CollisionHandlerEvent* collision_handler_Ptr )
                              : NodePath( new CollisionNode( chen.get_name() ) ),
                                ReferenceCount(),
                                _collision_handler_Ptr( collision_handler_Ptr ),
                                _events( chen._events ),
                                _is_from_collider( chen._is_from_collider )
 {
    // copy again pattern
    size_t pattern_size( chen._collision_handler_Ptr->get_num_again_patterns() );
    for ( size_t i(0); i < pattern_size; ++i )
        _collision_handler_Ptr->add_again_pattern( chen._collision_handler_Ptr->get_again_pattern( i ) );
    
    // copy in pattern
    pattern_size = chen._collision_handler_Ptr->get_num_in_patterns();
    for ( size_t i(0); i < pattern_size; ++i )
        _collision_handler_Ptr->add_in_pattern( chen._collision_handler_Ptr->get_in_pattern( i ) );
    
    // copy out pattern
    pattern_size = chen._collision_handler_Ptr->get_num_out_patterns();
    for ( size_t i(0); i < pattern_size; ++i )
        _collision_handler_Ptr->add_out_pattern( chen._collision_handler_Ptr->get_out_pattern( i ) );
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  * To activate a collider, use reparent_to() to attach this node to a NodePath.
  * To deactivate a collider, use detach_node().
  * CAVEAT: Avoid invoking this constructor for classes inheriting from this class;
  * use the above constructor instead and set the second argument (i.e. initialize)
  * to false.
  *
  * @param (const std::string&) name - name for this CollisionHandlerEventNode
  */
 template <typename T>
 CollisionHandlerEventNode<T>::CollisionHandlerEventNode( const std::string& name, bool is_from_collider )
                              : NodePath( new CollisionNode( name ) ),
                                ReferenceCount(),
                                _collision_handler_Ptr( new CollisionHandlerEvent() ),
                                _is_from_collider( is_from_collider )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 CollisionHandlerEventNode<T>::CollisionHandlerEventNode( const CollisionHandlerEventNode<T>& chen )
                              : NodePath( chen ),
                                ReferenceCount(),
                                _collision_handler_Ptr( chen._collision_handler_Ptr ),
                                _events( chen._events ),
                                _is_from_collider( chen._is_from_collider )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 CollisionHandlerEventNode<T>::~CollisionHandlerEventNode()
 {
    // Remove the reference to the PandaNode of this Collider in the global traverser (for from colliders)
    CollisionHandlerEventNode<T>::turn_off_from_collision(); // or actually this may be unnecessary
    
    // Remove child-parent PointerTo references
    // and detach all NodePaths below this NodePath
    NodePathCollection collection( NodePath::get_children() );
    for ( size_t i( 0 ); i < collection.get_num_paths(); ++i )
    {
        NodePath node_path( collection.get_path( i ) );
        collection.add_paths_from( node_path.get_children() );
        node_path.detach_node();
    }
    
    // Detach this NodePath
    CollisionHandlerEventNode<T>::detach_node();
 }
 
 /**
  * operator=() copies the content of the specified CollisionHandlerEventNode 
  * to this CollisionHandlerEventNode.
  *
  * @param (const CollisionHandlerEventNode<T>& chen )
  * @return CollisionHandlerEventNode<T>&
  */
 /*template <typename T>
 inline CollisionHandlerEventNode<T>& CollisionHandlerEventNode<T>::operator=( const CollisionHandlerEventNode<T>& chen )
 {
    NodePath::operator=( chen );
    //ReferenceCount::operator=( chen );
    _collision_handler_Ptr = chen._collision_handler_Ptr;
    _events = chen._events;
    return *this;
 }*/
 
 /**
  * add_event() adds an event to be handled for this CollisionHandlerEventNode with
  * the specified event_name. True is returned if the event_name has not already 
  * been specified for the global event handler; otherwise, false is returned 
  * and the event is not added to the global event handler.
  *
  * @param (const std::string&) event_name
  * @param (EventHandler::EventFunction*) event_function
  * @return bool
  */
 template <typename T>
 inline bool CollisionHandlerEventNode<T>::add_event( const std::string& event_name, EventHandler::EventFunction* event_function )
 {
    if ( !NodePath::is_empty() )
    {
        bool is_newly_added( CollisionHandlerEventNode<T>::add_global_event( event_name, event_function ) );
        
        // FROM_COLLISION_EVENT_NAME and FROM_COLLISION_EVENT_FUNCTION pair is reserved for from collisions
        if ( is_newly_added && (event_name != FROM_COLLISION_EVENT_NAME || event_function != FROM_COLLISION_EVENT_FUNCTION) )
            _events.push_back( new CollisionHandlerEventNode<T>::CollisionEvent( event_name, event_function ) );
        
        return is_newly_added;
    }
    
    return false;
 }
 
 /**
  * clear_events() removes all collision events added to this CollisionHandlerEventNode.
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::clear_events()
 {
    /*size_t event_index( _events.size() );
    while ( event_index > 0 )
    {
        --event_index;
        
        CollisionHandlerEventNode<T>::CollisionEvent& c_event( _events[ event_index ] );
        global::_framework.get_event_handler().remove_hook( c_event.name, c_event.function );
        //global::_framework.get_event_handler().remove_hooks( c_event.name );
    }*/
    
    _events.clear();
 }
 
 /**
  * get_copy() returns a pointer to a CollisionHandlerEventNode that hard copies this
  * CollisionHandlerEventNode.
  *
  * @return PT(CollisionHandlerEventNode<T>)
  */
 template <typename T>
 inline PT(CollisionHandlerEventNode<T>) CollisionHandlerEventNode<T>::get_copy() const
 {
    return new CollisionHandlerEventNode<T>( *this, new CollisionHandlerEvent() );
 }
 
 /**
  * remove_event() removes all events to be handled for this CollisionHandlerEventNode with
  * the specified event_name. If the event_name has been specified for this CollisionHandlerEventNode, 
  * the collision event is removed and true is returned; otherwise, false is returned.
  *
  * @param (const std::string&) event_name
  * @param (EventHandler::EventFunction*) event_function
  * @return bool
  */
 template <typename T>
 inline bool CollisionHandlerEventNode<T>::remove_events( const std::string& event_name )
 {
    bool removed( false );
    
    size_t event_index( _events.size() );
    while ( event_index > 0 )
    {
        //--event_index;
        PT(CollisionHandlerEventNode<T>::CollisionEvent)& c_event_Ptr( _events[ --event_index ] );
        
        // if the event_name is found
        if ( c_event_Ptr->_name == event_name )
        {
            // remove the event name
            PT(CollisionHandlerEventNode<T>::CollisionEvent)& last_event_Ptr( _events[ _events.size() - 1 ] );
            SWAP( c_event_Ptr, last_event_Ptr, PT(CollisionHandlerEventNode<T>::CollisionEvent) );
            _events.pop_back();
            removed = true;
        }
    }
    
    return removed;
 }
 
 /**
  * reparent_to() attaches this collision handler node to the specified NodePath.
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::reparent_to( NodePath& node_path )
 {
    // if the collision handler has not been initialized yet
    //if ( _collision_handler_Ptr == NULL )
        //initialize_collision_handler();
    
    if ( _is_from_collider )
        CollisionHandlerEventNode<T>::turn_on_from_collision();
    
    // Top: parent
    // 2nd: *this
    NodePath::reparent_to( node_path );
 }
 
 /**
  * detach_node() makes collision impossible for this node by detaching this collision handler.
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::detach_node()
 {
    bool temp( _is_from_collider );
    CollisionHandlerEventNode<T>::turn_off_from_collision();
    _is_from_collider = temp;
    NodePath::detach_node();
 }
 
 /**
  * turn_on_from_collision() makes this node become a "from" collider.
  * This node is already naturally an "into" collider.
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::turn_on_from_collision()
 {
    nassertv( _collision_handler_Ptr != NULL );
    
    CollisionTraverser& collision_traverser( global::_world_Ptr->get_map_area()->get_collision_traverser() );
    if ( !collision_traverser.has_collider( *this ) )
    {
        _is_from_collider = true;
        collision_traverser.add_collider( *this, _collision_handler_Ptr );
    }
 }
 
 /**
  * turn_off_from_collision() makes this node become strictly an "into" collider.
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::turn_off_from_collision()
 {
    //if ( _is_from_collider )
    //{
        _is_from_collider = false;
        global::_world_Ptr->get_map_area()->get_collision_traverser().remove_collider( *this );
    //}
 }
 
 /**
  * collision_node() returns a pointer to the CollisionNode represented by this CollisionHandlerEventNode.
  *
  * @return CollisionNode*
  */
 template <typename T>
 inline CollisionNode* CollisionHandlerEventNode<T>::collision_node()
 {
    return reinterpret_cast<CollisionNode*>( NodePath::node() );
 }
 
 /** PRIVATE FUNCTIONS **/
 
 /**
  * initialize_collision_handler() initializes the collision handling for this CollisionHandlerEventNode.
  * A CollisionHandlerEventNode is initialized to be an "into" and "from" collider.
  *
  * @param (const std::string&) collider_name
  */
 template <typename T>
 inline void CollisionHandlerEventNode<T>::initialize_collision_handler()
 {
    //PT(CollisionHandlerEvent) event_handler_Ptr( new CollisionHandlerEvent() );
    //_collision_handler_Ptr = event_handler_Ptr;
    //_collision_handler_Ptr = new CollisionHandlerEvent();
    
    // assert the collision handler has been allocated
    //nassertv( !_collision_handler_Ptr.is_null() );
    
    // add_collider( collision_solid_node_path, node_path_effected_by_collision )
    //event_handler_Ptr->add_collider( *this, parent );
    
    //CollisionHandlerEventNode<T>::turn_on_from_collision();
    //_is_from_collider = true;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 /**
  * add_global_event() adds an event to be handled for the specified event_name. True is returned if the 
  * event_name has not already been specified for the global event handler; otherwise, false 
  * is returned and the event is not added to the global event handler.
  *
  * @param (const std::string&) event_name
  * @param (EventHandler::EventFunction*) event_function
  * @return bool
  */
 template <typename T>
 inline bool CollisionHandlerEventNode<T>::add_global_event( const std::string& event_name, EventHandler::EventFunction* event_function )
 {
    return global::_framework.get_event_handler().add_hook( event_name, event_function );
 }
 
 /**
  * remove_global_event() removes an event having the specified event_name and event_function. 
  * True is returned if the event is removed; otherwise, false is returned because the event 
  * is currently not handled by (i.e. exists in the) global event handler.
  *
  * @param (const std::string&) event_name
  * @param (EventHandler::EventFunction*) event_function
  * @return bool
  */
 template <typename T>
 inline bool CollisionHandlerEventNode<T>::remove_global_event( const std::string& event_name, EventHandler::EventFunction* event_function )
 {
    return global::_framework.get_event_handler().remove_hook( event_name, event_function );
 }
 
 #undef FROM_COLLISION_EVENT_NAME
 
 #endif // COLLISION_HANDLER_EVENT_NODE_H